/* ====================================================================
 * Copyright (c) 2002 Vincent Partington and Erwin Bolwidt.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY VINCENT PARTINGTON ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL VINCENT PARTINGTON OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 */

package org.klomp.eclipse.formatjava;

import java.util.MissingResourceException;
import java.util.ResourceBundle;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPluginDescriptor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.plugin.AbstractUIPlugin;

/**
 * The main plugin class for the "Format .java" plugin.
 * 
 * @author Vincent Partington &lt;vinny@klomp.org&gt;
 * @author Erwin Bolwidt &lt;erwin@klomp.org&gt;
 */
public class FormatJavaPlugin extends AbstractUIPlugin {

	/**
	 * The shared plugin instance.
	 */
	private static FormatJavaPlugin plugin;

	/**
	 * The resource bundle for this plugin.
	 */
	private ResourceBundle resourceBundle;

	/**
	 * The constructor.
	 */
	public FormatJavaPlugin(IPluginDescriptor descriptor) {
		super(descriptor);
		plugin = this;
		resourceBundle = ResourceBundle.getBundle("plugin");
	}

	/**
	 * Returns the shared instance.
	 */
	public static FormatJavaPlugin getDefault() {
		return plugin;
	}

	/**
	 * Returns the plugin's resource bundle,
	 */
	public ResourceBundle getResourceBundle() {
		return resourceBundle;
	}

	/**
	 * Returns the string from the plugin's resource bundle,
	 * or 'key' if not found.
	 */
	public String getResourceString(String key) {
		ResourceBundle bundle = getResourceBundle();
		try {
			return bundle.getString(key);
		} catch (MissingResourceException e) {
			return key;
		}
	}

	/**
	 * The key of the resource bundle entry with the english text
	 * "Cannot replace contents of document".
	 */
	public static final String CANNOT_REPLACE_CONTENTS_RESOURCE_KEY =
		"cannot.replace.contents";

	/**
	 * The key of the resource bundle entry with the english text
	 * "An error occurred while processing the files".
	 */
	public static final String ERROR_WHILE_PROCESSING_RESOURCE_KEY =
		"error.while.processing";

	/**
	 * The key of the resource bundle entry with the english text
	 * "Formatting".
	 */
	public static final String FORMATTING_RESOURCE_KEY = "formatting";

	/**
	 * Reports an exception to the user.
	 * 
	 * @param shell the <code>Shell</code> to which the error should be reported
	 * @param messageResourceKey the resource key of the internationalized error
	 * message that should accompany the exception
	 * @param rootCause the actual exception that occurred.
	 */
	public void reportException(
		Shell shell,
		String messageResourceKey,
		Throwable rootCause) {
		//
		IStatus status = createStatus(messageResourceKey, rootCause);
		ErrorDialog.openError(
			shell,
			plugin.getDescriptor().getLabel(),
			status.getMessage(),
			status);

	}

	/**
	 * Reports a <code>CoreException</code> to the user.
	 * 
	 * @param shell the <code>Shell</code> to which the error should be reported
	 * @param messageResourceKey the resource key of the internationalized error
	 * message that should accompany the exception
	 * @param rootCause the actual exception that occurred.
	 */
	public void reportException(
		Shell shell,
		String messageResourceKey,
		CoreException rootCause) {
		//
		ErrorDialog.openError(
			shell,
			getDescriptor().getLabel(),
			getResourceString(messageResourceKey),
			rootCause.getStatus());
	}

	/**
	 * Throws a <code>CoreException</code> that wraps another
	 * type of exception.
	 * 
	 * @param messageResourceKey the resource key of the internationalized error
	 * message that should accompany the exception
	 * @param rootCause the actual exception that occurred.
	 */
	public void throwCoreException(
		String messageResourceKey,
		Throwable rootCause)
		throws CoreException {
		//
		throw createCoreException(messageResourceKey, rootCause);
	}

	/**
	 * Creates a <code>CoreException</code> that wraps another
	 * type of exception.
	 * 
	 * @param messageResourceKey the resource key of the internationalized error
	 * message that should accompany the exception
	 * @param rootCause the actual exception that occurred.
	 */
	public CoreException createCoreException(
		String messageResourceKey,
		Throwable rootCause) {
		//
		return new CoreException(createStatus(messageResourceKey, rootCause));
	}

	/**
	 * Creates an <code>IStatus</code> object for an exception.
	 * 
	 * @param messageResourceKey the resource key of the internationalized error
	 * message that should accompany the exception
	 * @param rootCause the actual exception that occurred.
	 */
	public IStatus createStatus(
		String messageResourceKey,
		Throwable rootCause) {
		//
		return new Status(
			IStatus.ERROR,
			getDescriptor().getUniqueIdentifier(),
			IStatus.OK,
			getResourceString(messageResourceKey),
			rootCause);
	}

}
