/* ====================================================================
 * Copyright (c) 2002 Vincent Partington and Erwin Bolwidt.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY VINCENT PARTINGTON ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL VINCENT PARTINGTON OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 */

package org.klomp.eclipse.formatjava;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jdt.core.ICodeFormatter;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.ToolFactory;
import org.eclipse.jdt.ui.JavaUI;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.source.IAnnotationModel;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.part.FileEditorInput;
import org.eclipse.ui.texteditor.IDocumentProvider;

/**
 * The operation of formatting an array of compilation units (Java source
 * files), and making sure that any open editors on files stay in sync
 * with the results. Any changed files that weren't changed in an open
 * editor are directly saved to disk. For open files, the editor shows
 * the result of the format operation, but the file is not changed on
 * disk.
 *  
 * @author Vincent Partington &lt;vinny@klomp.org&gt;
 * @author Erwin Bolwidt &lt;erwin@klomp.org&gt;
 */
public class FormatJavaOperation implements IWorkspaceRunnable {

	private ICompilationUnit[] units;

	/**
	 * Constructs of <code>FormatJavaOperation</code> object.
	 * 
	 * @param units an array of Java Compilcation Units to compile
	 */
	public FormatJavaOperation(ICompilationUnit[] units) {
		this.units = units;
	}

	/**
	 * Runs the operation.
	 * 
	 * @param monitor the <code>IProgressMonitor</code> to which the
	 * formatting progress should be reported
	 */
	public void run(IProgressMonitor monitor) throws CoreException {
		monitor.beginTask(
			FormatJavaPlugin.FORMATTING_RESOURCE_KEY,
			units.length);
		processJavaFiles(monitor);
		monitor.done();
	}

	/**
	 * Process an array of Java files.
	 * 
	 * @param monitor the <code>IProgressMonitor</code> to which the
	 * formatting progress should be reported
	 */
	private void processJavaFiles(IProgressMonitor monitor)
		throws CoreException {
		for (int i = 0; i < units.length; i++) {
			ICompilationUnit cu = units[i];
			ICompilationUnit working =
				(ICompilationUnit) cu.findSharedWorkingCopy(
					JavaUI.getBufferFactory());
			if (working != null)
				cu = working;

			boolean unsavedChanges = cu.hasUnsavedChanges();

			monitor.subTask(units[i].getPath().toString());
			processJavaFile(monitor, cu, !unsavedChanges);
			monitor.worked(1);
		}
	}

	/**
	 * Process one Java file.
	 * 
	 * @param monitor the <code>IProgressMonitor</code> to which the
	 * formatting progress should be reported
	 * @param cu the Java file to format
	 * @param save if <code>true</code> the file is not open in an editor
	 * and should be saved. Otherwise, the formatted file will not be saved
	 */
	private void processJavaFile(
		final IProgressMonitor monitor,
		final ICompilationUnit cu,
		final boolean save)
		throws CoreException {
		// get file to process
		ICompilationUnit orig = cu;
		if (cu.isWorkingCopy())
			orig = (ICompilationUnit) cu.getOriginalElement();

		IFile file = (IFile) orig.getUnderlyingResource();

		final IDocumentProvider docProvider = JavaUI.getDocumentProvider();
		final IFileEditorInput input = new FileEditorInput(file);

		docProvider.connect(input);

		try {
			final IDocument document = docProvider.getDocument(input);
			final IAnnotationModel annotationModel =
				docProvider.getAnnotationModel(input);
			annotationModel.connect(document);
			try {
				IWorkspaceRunnable action = new IWorkspaceRunnable() {
					public void run(IProgressMonitor monitor)
						throws CoreException {
						try {
							int length = document.getLength();
							String source = document.get(0, length);
							String formatted = formatJava(source);
							if (!formatted.equals(source)) {
								document.replace(0, length, formatted);
								docProvider.aboutToChange(input);
								try {
									if (save
										|| docProvider.mustSaveDocument(input)) {
										//
										docProvider.saveDocument(
											monitor,
											input,
											document,
											true);
									}
								} finally {
									docProvider.changed(input);
								}
							}
						} catch (BadLocationException exc) {
							FormatJavaPlugin.getDefault().throwCoreException(
								FormatJavaPlugin
									.CANNOT_REPLACE_CONTENTS_RESOURCE_KEY,
								exc);
						}
					}
				};

				ResourcesPlugin.getWorkspace().run(action, monitor);
			} finally {
				annotationModel.disconnect(document);
			}
		} finally {
			docProvider.disconnect(input);
		}
	}

	/**
	 * Formats Java source code.
	 * 
	 * @param source the Java source code to format
	 * @return the formatted Java source code
	 */
	private static String formatJava(String source) {
		ICodeFormatter formatter = ToolFactory.createCodeFormatter();

		return formatter.format(source, 0, null, null);
	}

}
