/* ====================================================================
 * Copyright (c) 2002 Vincent Partington and Erwin Bolwidt.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY VINCENT PARTINGTON ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL VINCENT PARTINGTON OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 */

package org.klomp.eclipse.formatjava;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IObjectActionDelegate;
import org.eclipse.ui.IWorkbenchPart;

/**
 * Action code for the "Format .java files" plugin.
 * 
 * @author Vincent Partington &lt;vinny@klomp.org&gt;
 * @author Erwin Bolwidt &lt;erwin@klomp.org&gt;
 */
public class FormatJavaAction implements IObjectActionDelegate {

	/**
	 * The part in which this action is run.
	 */
	private IWorkbenchPart part;

	/**
	 * The most currently known selection.
	 */
	private IStructuredSelection selection;

	/**
	 * Called by the workbench to set the active part for the delegate.
	 * 
	 * @see org.eclipse.ui.IObjectActionDelegate#setActivePart
	 */
	public void setActivePart(IAction action, IWorkbenchPart targetPart) {
		this.part = targetPart;
	}

	/**
	 * Called by the workbench when the selection in a file selection window
	 * changes, and the workbench thinks that the selected files may be
	 * applicable to this action, as defined in the plugin descriptor.
	 * 
	 * @see org.eclipse.ui.IActionDelegate#selectionChanged
	 */
	public void selectionChanged(IAction action, ISelection selection) {
		if (selection instanceof IStructuredSelection)
			this.selection = (IStructuredSelection) selection;
		else
			this.selection = null;
	}

	/**
	 * Called by the workbench to perform this action. It calls the
	 * perform method for the real work and reports the errors generated
	 * by the perform method to the user.
	 * 
	 * @see org.eclipse.ui.IActionDelegate#run
	 */
	public void run(IAction action) {
		Shell shell = part.getSite().getShell();
		try {
			perform(shell);
		} catch (CoreException exc) {
			FormatJavaPlugin.getDefault().reportException(
				shell,
				FormatJavaPlugin.ERROR_WHILE_PROCESSING_RESOURCE_KEY,
				exc);
		} catch (OperationCanceledException exc) {
			// don't display an error message when the operation is cancelled.
		} catch (RuntimeException exc) {
			FormatJavaPlugin.getDefault().reportException(
				shell,
				FormatJavaPlugin.ERROR_WHILE_PROCESSING_RESOURCE_KEY,
				exc);
		}
	}

	/**
	 * The actual processing, called by the run method.
	 * 
	 * @exception CoreException if something went wrong
	 */
	private void perform(Shell shell) throws CoreException {
		// no selection?
		if (selection == null)
			return;

		// determine the files to process
		final ICompilationUnit[] units = expandSelection(selection);

		// any files to process?
		if (units.length == 0)
			return;

		// create a progress monitor
		ProgressMonitorDialog dialog = new ProgressMonitorDialog(shell);
		try {
			// run FormatJavaOperation 'in' the progress monitor
			dialog.run(false, true, new IRunnableWithProgress() {
				public void run(IProgressMonitor monitor)
					throws InvocationTargetException, InterruptedException {
					try {
						ResourcesPlugin.getWorkspace().run(
							new FormatJavaOperation(units),
							monitor);
					} catch (OperationCanceledException e) {
						throw new InterruptedException(e.getMessage());
					} catch (CoreException e) {
						throw new InvocationTargetException(e);
					}
				}
			});
		} catch (InvocationTargetException exc) {
			Throwable targetException = exc.getTargetException();
			if (targetException instanceof CoreException) {
				throw (CoreException) targetException;
			} else {
				FormatJavaPlugin.getDefault().throwCoreException(
					FormatJavaPlugin.ERROR_WHILE_PROCESSING_RESOURCE_KEY,
					targetException);
			}
		} catch (InterruptedException e) {
			// ignore InterruptedExceptions that are caused by the user
			// cancelling out of a long running operation
		}

	}

	/**
	 * Scans the selection for Java files to format.
	 * 
	 * @param selection the selection to scan
	 *
	 * @return the Java files (compilation units) to format
	 * 
	 * @exception CoreException if something went wrong
	 */
	private static ICompilationUnit[] expandSelection(ISelection selection)
		throws CoreException {
		//
		if (!(selection instanceof IStructuredSelection))
			return new ICompilationUnit[0];
		IStructuredSelection ss = (IStructuredSelection) selection;

		List result = new ArrayList();
		expandObjects(ss.toArray(), result);

		return (ICompilationUnit[]) result.toArray(
			new ICompilationUnit[result.size()]);
	}

	/**
	 * Scans an array of Objects for Java files to format.
	 * 
	 * @param objects the array to scan
	 * @param result the <code>Collection</code> to which found
	 * files should be added.
	 * 
	 * @exception CoreException if something went wrong
	 */
	private static void expandObjects(Object[] objects, Collection result)
		throws CoreException {
		//
		for (int i = 0; i < objects.length; i++) {
			Object o = objects[i];
			if (o instanceof ICompilationUnit) {
				// add a Java file to the list of files to process
				result.add(o);
			} else if (o instanceof IPackageFragment) {
				// add all Java files in a package fragment to the list
				IPackageFragment pack = (IPackageFragment) o;
				result.addAll(Arrays.asList(pack.getCompilationUnits()));
			} else if (o instanceof IPackageFragmentRoot) {
				// recursively call this function to scan all package fragments
				// in the root for Java files
				IPackageFragmentRoot fragRoot = (IPackageFragmentRoot) o;
				expandObjects(fragRoot.getChildren(), result);
			}
		}
	}

}
